<?php
defined('BASEPATH') or exit('No direct script access allowed');
/**
 * @author Scansio Qui <scansioquielom@gmail.com>
 * @since 1.0.0
 * Last update: 01/10/2022
 */
class Scansio_db_man extends Ci_Model
{
	private $conn = NULL;
	private $last_error;

	/**
	 * @param string $dbName Optional database name as default database to be used in SQL transactions in this object.
	 */
	public function __construct($dbName = NULL)
	{
		try {
			set_error_handler(function ($error) {
				$this->setLastError($error);
			});
			$this->conn = new mysqli($this->db->hostname, $this->db->username, $this->db->password);
			if ($dbName !== NULL) {
				$dbName = htmlentities($dbName, ENT_QUOTES);
				$this->createDatabase($dbName, true);
			} else {
				if (!empty($this->db->database))
					$this->createDatabase($this->db->database, true);
			}
			if ($this->conn === NULL || $this->conn->connect_error) {
				die("Error in connection: " . $this->conn->connect_error);
			}
		} catch (Exception $e) {
			$this->setLastError($e);
			die();
		}
	}

	private function setLastError($error)
	{
		log_message('debug', $error);
		$this->last_error = $error;
	}

	/**
	 * @return string
	 */
	public function getLastError()
	{
		return $this->last_error;
	}

	/**
	 * Create database.
	 * @param string $dbName The name of the database to create.
	 * @param bool $connectDatabase Selects the default database for database queries Selects the default database to be used when performing queries against the database connection.
	 * @return bool Returns true upon success false otherwise.
	 */
	public function createDatabase($dbName, $connectDatabase = false)
	{
		$dbName = ($dbName === NULL || ($dbName = trim($dbName)) === "") ? die() : htmlentities($dbName);
		$sql = "CREATE DATABASE IF NOT EXISTS " . $dbName . ";";
		try {
			if ($this->conn->query($sql)) {
				if ($connectDatabase) {
					$this->conn->select_db($dbName);
				}
				return true;
			}
		} catch (Exception $e) {
			$this->setLastError($e);
		}
		return false;
	}

	public function __destruct()
	{
		if ($this->conn !== NULL && $this->conn->close())
			;
	}

	/**
	 * This create's table with the name `$tableName`, and an associative array `$colums`
	 * using its key as column names and its value as column properties.
	 * Example:
	 *  `$columns = array(
	 *      "id" => "INT(10) UNSIGNED AUTO_INCREMENT PRIMARY KEY",
	 *      "firstname" => "VARCHAR(30) NOT NULL",
	 *      "lastname" => "VARCHAR(30) NOT NULL",
	 *      "email" => "VARCHAR(30) NOT NULL",
	 *      "reg_date" => "TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP"
	 *  );`
	 * @param string $tableName Table to create.
	 * @param array $columns Columns, Associative array its key as column name it value as column properties.
	 * @param string|null $dbName Optional database name if transacting in another database other than default database.
	 * @return bool Returns true upon success false otherwise.
	 */
	public function createTable($tableName, array $columns, $dbName = NULL)
	{
		$sql = "CREATE TABLE IF NOT EXISTS " . $this->getPreparedDbName($dbName) . "`" . htmlentities($tableName, ENT_QUOTES) . '` (';
		foreach ($columns as $key => $value) {
			$sql .= '`' . htmlentities($key, ENT_QUOTES) . '` ' . htmlentities($value, ENT_QUOTES) . ', ';
		}
		$sql = substr($sql, 0, -2) . ');';
		try {
			if ($this->conn->query($sql) === TRUE) {
				return true;
			}
		} catch (Exception $e) {
			$this->setLastError($e);
		}
		return false;
	}

	/**
	 * Checks and format the database name.
	 *
	 * @param string|null $dbName The database name.
	 * @return string Returns formatted database name.
	 */
	private function getPreparedDbName($dbName)
	{
		return ($dbName === NULL || ($dbName = trim($dbName)) == "") ? "" : '`' . htmlentities($dbName, ENT_QUOTES) . '`.';
	}

	/**
	 * Read single line
	 *
	 * @param string $tablename The table to get the result.
	 * @param string $field
	 * @param string $filter Criteria of which to select some result set.
	 * @param string|null $dbName Optional database name if transacting in another database other than default database.
	 * @return array|bool|null Returns and associative array of a single row or null if there were no match or false error.
	 */
	public function select($tablename, $field, $filter, $dbName = NULL)
	{
		try {
			$sql = $this->getQueryForSelector($tablename, $field, $filter, $dbName);
			if ($result_set = $this->conn->query($sql)) {
				return ($field == "*" || $field == '' || strstr($field, ",")) ? $result_set->fetch_assoc() : $result_set->fetch_assoc()[$field];
			}
		} catch (Exception $e) {
			$this->setLastError($e);
		}
		return false;
	}

	/**
	 * Helper method that returns query string for selectors.
	 * @param string $tableName The table to get the result.
	 * @param string $field fields to select
	 * @param string $filter Criteria of which to select some result set.
	 * @param string $dbName Optional database name if transacting in another database other than default database.
	 * @return string Returns a two-dimensional array of row or null if there is no result in the table or false on error.
	 */
	private function getQueryForSelector($tableName, $field, $filter, $dbName = NULL)
	{
		if ($field && strstr($field, ",")) {
			$l = "";
			foreach (explode(",", $field) as $i) {
				$l .= "`" . htmlentities(trim($i), ENT_QUOTES) . "`,";
			}
			$field = substr($l, 0, -1);
		} else if (($field = trim($field)) === "" || $field === "*") {
			$field = "*";
		} else {
			$field = "`" . htmlentities($field, ENT_QUOTES) . "`";
		}
		if ($tableName && strstr($tableName, ",")) {
			$l = "";
			foreach (explode(",", $tableName) as $i) {
				$l .= "`" . htmlentities(trim($i), ENT_QUOTES) . "`,";
			}
			$tableName = substr($l, 0, -1);
		} else if (($tableName = trim($tableName)) === "" || $tableName === "*") {
			$tableName = "*";
		} else {
			$tableName = "`" . htmlentities($tableName, ENT_QUOTES) . "`";
		}
		return "SELECT $field FROM " . $this->getPreparedDbName($dbName) . " $tableName $filter;";
	}

	public function escapeQuery($filter)
	{
		if ($filter && !(trim($filter) == '')) {
			//$filter = str_replace(";", '&semi;', $filter);
			$filter = str_replace("'", '&apos;', $filter);
			$filter = str_replace('"', '&quot;', $filter);
			$filter = str_replace('`', '&grave;', $filter);
		} else
			$filter = '';
		return $filter;
	}

	public function parseHtmlEntity($filter)
	{
		if ($filter && !(trim($filter) == '')) {
			//$filter = str_replace('&semi;', ";", $filter);
			$filter = str_replace('&apos;', "'", $filter);
			$filter = str_replace('&quot;', '"', $filter);
			$filter = str_replace('&grave;', '`', $filter);
		} else
			$filter = '';
		return $filter;
	}

	/**
	 * Select group line
	 * @param string $tableName The table to get the result.
	 * @param string $field The fields to select
	 * @param string $filter Criteria of which to select some result set.
	 * @param string|null $dbName Optional database name if transacting in another database other than default database.
	 * @return array|bool|null Returns a two-dimensional array of row or null if there is no result in the table or false on error.
	 */
	public function selectGroup($tableName, $field = '', $filter = '', $dbName = NULL)
	{
		try {
			$sql = $this->getQueryForSelector($tableName, $field, $filter, $dbName);
			if ($result_set = $this->conn->query($sql)) {
				if ($result_set->num_rows <= 0) {
					return [];
				}
				$result = [];
				while (($row = $result_set->fetch_assoc())) {
					$result[] = $row;
				}
				return $result;
			}
		} catch (Exception $e) {
			$this->setLastError($e);
		}
		return [];
	}

	/**
	 * Delete from table
	 *
	 * @param string $tablename The name of table to insert data to.
	 * @param string $filter Criteria of which to delete from the table.
	 * @param string|null $dbName Optional database name if transacting in another database other than default database.
	 * @return bool Returns true upon success false otherwise.
	 */
	public function delete($tablename, $filter, $dbName = NULL)
	{
		$sql = "DELETE FROM " . $this->getPreparedDbName($dbName) . "`" . htmlentities($tablename, ENT_QUOTES) . '` ' . $filter . ';';
		try {
			if ($this->conn->query($sql)) {
				return true;
			}
		} catch (Exception $e) {
			$this->setLastError($e);
		}
		return false;
	}

	/**
	 * Update table.
	 * Note: An associative array `$data` contains key as column name and value as column value.
	 * Example: If we want to execute a query like this
	 *          `"UPDATE myGuests SET firstname="Emma", lastname="Elom", email="com.syber.emma@gmail.com" WHERE id=2;"`
	 *          Thus, the `$update` will contain the update value
	 *          `$update = array(
	 *              "firstname" => "Emma",
	 *              "lastname" => "Elom",
	 *              "email" => "com.syber.emma@gmail.com"
	 *           ); `
	 *
	 *           Filter will be:
	 *           `$filter = "WHERE id=2"`
	 *          This means that the columns that march `$filter` criteria should be updated.
	 * @param string $tableName The name of table to insert data to.
	 * @param array $update Associative array which it key is the column name and it value is the column value which contains the update.
	 * @param string $filter Criteria of which to update the columns.
	 * @param string|null $dbName Optional database name if transacting in another database other than default database.
	 * @return bool Returns true upon success false otherwise.
	 */
	public function update($tableName, array $update, $filter, $dbName = NULL)
	{
		$sql = "UPDATE " . $this->getPreparedDbName($dbName) . "`" . htmlentities($tableName, ENT_QUOTES) . '` SET ';
		foreach ($update as $columnName => $updateValue) {
			//Checking for the supported value returns false if not supported
			if (is_int($updateValue) || is_float($updateValue))
				;
			else if (is_string($updateValue)) {
				$updateValue = $this->escapeQuery($updateValue);
			} else
				return false;
			//Sanitizing and concatenating the keys and values to SQL statement
			$sql .= '`' . htmlentities($columnName, ENT_QUOTES) . '` = "' . $updateValue . '", ';
		}
		//Removing the last comma concatenated in the loop above
		$sql = substr($sql, 0, -2) . ' ' . $filter . ';';
		try {
			if ($this->conn->query($sql)) {
				return true;
			}
		} catch (Exception $e) {
			$this->setLastError($e);
		}
		return false;
	}

	/**
	 * Insert data into table.
	 * Note: An associative array `$data` contains key as column name and value as column value.
	 * Example: If we want to execute a query like this:
	 *          `"INSERT INTO myGuests (firstname, lastname, email) VALUES ("Emma", "Elom", "com.syber.emma@gmail.com");"`
	 *          Thus;
	 *          `$data = array(
	 *              "firstname" => "Emma",
	 *              "lastname" => "Elom",
	 *              "email" => "com.syber.emma@gmail.com"
	 *           );`
	 *
	 * @param string $tablename The name of table to insert data to.
	 * @param array $data Associative array which it key is the column name and value the column value.
	 * @param string|null $dbName Optional database name if transacting in another database other than default database.
	 * @return bool Returns true upon success false otherwise.
	 */
	public function insert($tablename, array $data, $dbName = NULL)
	{
		$sql = "INSERT INTO " . $this->getPreparedDbName($dbName) . "`" . htmlentities($tablename, ENT_QUOTES) . '` (';
		$columns = '';
		$values = '';
		foreach ($data as $key => $value) {
			//Checking for the supported value returns false if not supported
			if (is_int($value) || is_float($value))
				; //Just leave the value if its of type int or float
			else if (is_string($value)) {
				$value = '"' . $this->escapeQuery($value) . '"';
			} else if ($value == NULL) {
				$value = '""';
			} else
				return false;
			$values .= $value . ', ';
			//Sanitizing and concatenating the key as columns name
			$columns .= '`' . htmlentities($key, ENT_QUOTES) . '`, ';
		}
		//Removing the last comma concatenated in the above loop
		//      |                      |
		$sql .= substr($columns, 0, -2) . ') VALUES (';
		$sql .= substr($values, 0, -2) . ');';
		try {
			if ($this->conn->query($sql)) {
				return true;
			}
		} catch (Exception $e) {
			$this->setLastError($e);
		}
		return false;
	}

	/**
	 * Returns number of rows in result set returned by sql query.
	 *
	 * @param string $tablename The name of table to select data from.
	 * @param string $filter Criteria of which to select some result set.
	 * @param string|null $dbName Optional database name if transacting in another database other than default database.
	 * @return int|string|bool Returns number of row count or string representation of row count or false if an error occured.
	 */
	public function rowCount($tablename, $filter = '', $dbName = NULL)
	{
		try {
			if ($result_set = $this->conn->query($this->getQueryForSelector($tablename, '', $filter, $dbName))) {
				return $result_set->num_rows;
			}
		} catch (Exception $e) {
			$this->setLastError($e);
		}
		return 0;
	}

	public function getLastErrorArray()
	{
		return $this->lastErrorArray;
	}

	public function rawQuery($query, $returnArray = true, $single = false)
	{
		try {
			if ($result_set = $this->conn->query($query)) {
				if ($result_set->num_rows < 1) {
					return false;
				}
				if (!$returnArray) {
					return $result_set;
				}
				if ($single) {
					return $result_set->fetch_assoc();
				}
				$result = [];
				while (($row = $result_set->fetch_assoc())) {
					$result[] = $row;
				}
				return $result;
			}
		} catch (Exception $e) {
			$this->setLastError($e);
		}
		return false;
	}
}